<?php

namespace app\controllers;

use Yii;
use app\models\Product;
use app\models\Category;
use yii\web\Controller;
use yii\web\NotFoundHttpException;
use yii\filters\VerbFilter;
use yii\data\ActiveDataProvider;
use yii\data\Sort;

class ProductController extends Controller
{
    public function behaviors()
    {
        return [
            'verbs' => [
                'class' => VerbFilter::class,
                'actions' => [
                    'delete' => ['POST'],
                ],
            ],
        ];
    }

    public function actionIndex()
    {
        $request = Yii::$app->request;

        // Получаем параметры фильтрации
        $categoryIds = $request->get('category', []);
        $priceMin = $request->get('price_min');
        $priceMax = $request->get('price_max');
        $sort = $request->get('sort', 'newest');
        $search = $request->get('search', '');

        // Основной запрос - только доступные товары
        $query = Product::getAvailableProducts()->with('category');

        // Поиск по названию
        if (!empty($search)) {
            $query->andWhere(['like', 'name', $search]);
        }

        // Фильтрация по категориям
        if (!empty($categoryIds) && !in_array('', $categoryIds)) {
            $query->andWhere(['in', 'category_id', $categoryIds]);
        }

        // Фильтрация по цене
        if ($priceMin !== null && $priceMin !== '') {
            $query->andWhere(['>=', 'price', (int)$priceMin]);
        }
        if ($priceMax !== null && $priceMax !== '') {
            $query->andWhere(['<=', 'price', (int)$priceMax]);
        }

        // Сортировка
        $query->orderBy($this->getSortOrder($sort));

        $dataProvider = new ActiveDataProvider([
            'query' => $query,
            'pagination' => [
                'pageSize' => 12,
            ],
            'sort' => false, // Отключаем стандартную сортировку, используем нашу
        ]);

        // Все категории для фильтра
        $categories = Category::find()->all();

        // Метки для сортировки
        $sortLabels = [
            'newest' => 'По умолчанию (новые)',
            'name_asc' => 'По названию (А-Я)',
            'name_desc' => 'По названию (Я-А)',
            'price_asc' => 'По цене (сначала дешевые)',
            'price_desc' => 'По цене (сначала дорогие)',
            'popular' => 'По популярности',
        ];

        return $this->render('index', [
            'dataProvider' => $dataProvider,
            'categories' => $categories,
            'selectedCategories' => $categoryIds,
            'priceMin' => $priceMin,
            'priceMax' => $priceMax,
            'sort' => $sort,
            'sortLabels' => $sortLabels,
            'search' => $search,
        ]);
    }

    public function actionView($id)
    {
        // Используем метод с проверкой доступности
        $product = Product::getAvailableProduct($id);

        if (!$product) {
            // Показываем более информативное сообщение
            throw new NotFoundHttpException('Товар не найден или временно недоступен. Вернитесь в каталог для выбора других товаров.');
        }

        // Похожие товары (из той же категории)
        $similarProducts = Product::getAvailableProducts()
            ->where(['category_id' => $product->category_id])
            ->andWhere(['!=', 'id', $id])
            ->limit(4)
            ->all();

        // Проверяем, есть ли товар в корзине пользователя
        $cartItem = null;
        if (!Yii::$app->user->isGuest) {
            $cartItem = \app\models\Cart::find()
                ->where([
                    'user_id' => Yii::$app->user->id,
                    'product_id' => $id
                ])
                ->one();
        }

        return $this->render('view', [
            'product' => $product,
            'similarProducts' => $similarProducts,
            'cartItem' => $cartItem,
        ]);
    }

    public function actionCategory($id)
    {
        $category = Category::findOne($id);
        if (!$category) {
            throw new NotFoundHttpException('Категория не найдена.');
        }

        $query = Product::getAvailableProducts()
            ->where(['category_id' => $id])
            ->orderBy(['id' => SORT_DESC]);

        $dataProvider = new ActiveDataProvider([
            'query' => $query,
            'pagination' => [
                'pageSize' => 12,
            ],
        ]);

        return $this->render('category', [
            'category' => $category,
            'dataProvider' => $dataProvider,
        ]);
    }

    public function actionSearch()
    {
        $request = Yii::$app->request;
        $q = $request->get('q', '');

        if (empty($q)) {
            return $this->redirect(['index']);
        }

        $query = Product::getAvailableProducts()
            ->andWhere(['or',
                ['like', 'name', $q],
                ['like', 'description', $q]
            ]);

        $dataProvider = new ActiveDataProvider([
            'query' => $query,
            'pagination' => [
                'pageSize' => 12,
            ],
        ]);

        return $this->render('search', [
            'query' => $q,
            'dataProvider' => $dataProvider,
        ]);
    }

    private function getSortOrder($sort)
    {
        $orders = [
            'newest' => ['id' => SORT_DESC],
            'name_asc' => ['name' => SORT_ASC],
            'name_desc' => ['name' => SORT_DESC],
            'price_asc' => ['price' => SORT_ASC],
            'price_desc' => ['price' => SORT_DESC],
            'popular' => ['counts' => SORT_ASC], // Меньше остаток = более популярный
        ];

        return $orders[$sort] ?? $orders['newest'];
    }

    // Простой метод для тестирования - создание тестовых данных
    public function actionSeed()
    {
        // Только для разработки
        if (YII_ENV_PROD) {
            throw new \yii\web\ForbiddenHttpException('Доступ запрещен');
        }

        // Убедимся, что категории существуют
        $categories = Category::find()->all();
        if (empty($categories)) {
            Yii::$app->session->setFlash('error', 'Сначала создайте категории');
            return $this->redirect(['category/index']);
        }

        // Создаем тестовые товары
        $testProducts = [];
        foreach ($categories as $category) {
            for ($i = 1; $i <= 3; $i++) {
                $testProducts[] = [
                    'name' => "Тестовый товар {$i} - {$category->name}",
                    'image' => "test{$i}.jpg",
                    'description' => "Описание тестового товара {$i} из категории {$category->name}",
                    'counts' => rand(10, 100),
                    'price' => rand(100, 1000),
                    'category_id' => $category->id,
                ];
            }
        }

        // Вставляем товары
        foreach ($testProducts as $productData) {
            $product = new Product();
            $product->attributes = $productData;
            $product->save();
        }

        Yii::$app->session->setFlash('success', 'Тестовые данные созданы');
        return $this->redirect(['index']);
    }
}