<?php

namespace app\controllers;

use Yii;
use app\models\Order;
use app\models\OrderItem;
use app\models\Cart;
use app\models\Product;
use yii\web\Controller;
use yii\web\NotFoundHttpException;
use yii\filters\VerbFilter;
use yii\filters\AccessControl;
use yii\data\ActiveDataProvider;

class OrderController extends Controller
{
    public function behaviors()
    {
        return [
            'access' => [
                'class' => AccessControl::class,
                'rules' => [
                    [
                        'allow' => true,
                        'roles' => ['@'], // Только авторизованные пользователи
                    ],
                ],
            ],
            'verbs' => [
                'class' => VerbFilter::class,
                'actions' => [
                    'delete' => ['POST'],
                    'cancel' => ['POST'],
                    'create' => ['GET', 'POST'], // РАЗРЕШАЕМ И GET, И POST
                    'update-status' => ['POST'],
                ],
            ],
        ];
    }

    public function actionIndex()
    {
        $query = Order::find()
            ->where(['user_id' => Yii::$app->user->id])
            ->orderBy(['created_at' => SORT_DESC]);

        $dataProvider = new ActiveDataProvider([
            'query' => $query,
            'pagination' => [
                'pageSize' => 10,
            ],
        ]);

        return $this->render('index', [
            'dataProvider' => $dataProvider,
        ]);
    }

    public function actionView($id)
    {
        $order = Order::find()
            ->where(['id' => $id, 'user_id' => Yii::$app->user->id])
            ->with(['orderItems.product', 'user'])
            ->one();

        if (!$order) {
            throw new NotFoundHttpException('Заказ не найден.');
        }

        return $this->render('view', [
            'order' => $order,
        ]);
    }

    public function actionCreate()
    {
        $request = Yii::$app->request;
        $userId = Yii::$app->user->id;

        // Получаем товары из корзины
        $cartItems = Cart::find()
            ->where(['user_id' => $userId])
            ->with('product')
            ->all();

        if (empty($cartItems)) {
            Yii::$app->session->setFlash('error', 'Ваша корзина пуста.');
            return $this->redirect(['cart/index']);
        }

        // Проверяем наличие товаров
        foreach ($cartItems as $item) {
            if ($item->product->counts < $item->quantity) {
                Yii::$app->session->setFlash('error',
                    "Товар '{$item->product->name}' недоступен в запрашиваемом количестве.");
                return $this->redirect(['cart/index']);
            }
        }

        $model = new Order();
        $model->user_id = $userId;
        $model->setStatusToNew();
        $model->created_at = date('Y-m-d H:i:s');

        if ($model->load($request->post())) {
            // Рассчитываем итоговую сумму
            $totalPrice = 0;
            foreach ($cartItems as $item) {
                $totalPrice += $item->quantity * $item->product->price;
            }
            $model->total_price = $totalPrice;

            // Добавляем стоимость доставки для курьера
            if ($model->isDeliveryMethodCourier()) {
                $model->total_price += 300; // Стоимость доставки
            }

            $transaction = Yii::$app->db->beginTransaction();
            try {
                if ($model->save()) {
                    // Создаем позиции заказа
                    foreach ($cartItems as $item) {
                        $orderItem = new OrderItem();
                        $orderItem->order_id = $model->id;
                        $orderItem->product_id = $item->product_id;
                        $orderItem->quantity = $item->quantity;
                        $orderItem->price_per_item = $item->product->price;

                        if (!$orderItem->save()) {
                            throw new \Exception('Ошибка сохранения позиции заказа: ' . print_r($orderItem->errors, true));
                        }

                        // Уменьшаем количество товара на складе
                        $product = $item->product;
                        $product->counts -= $item->quantity;
                        if (!$product->save()) {
                            throw new \Exception('Ошибка обновления количества товара: ' . print_r($product->errors, true));
                        }
                    }

                    // Очищаем корзину
                    Cart::deleteAll(['user_id' => $userId]);

                    $transaction->commit();

                    // Отправляем уведомление (опционально)
                    $this->sendOrderConfirmation($model);

                    Yii::$app->session->setFlash('success',
                        'Заказ успешно создан! Номер вашего заказа: ' . $model->id);

                    return $this->redirect(['view', 'id' => $model->id]);
                } else {
                    throw new \Exception('Ошибка сохранения заказа: ' . print_r($model->errors, true));
                }
            } catch (\Exception $e) {
                $transaction->rollBack();
                Yii::error($e->getMessage(), 'order');
                Yii::$app->session->setFlash('error', 'Ошибка при создании заказа: ' . $e->getMessage());
            }
        }

        // Рассчитываем итоговую сумму для отображения
        $totalPrice = 0;
        foreach ($cartItems as $item) {
            $totalPrice += $item->quantity * $item->product->price;
        }

        return $this->render('create', [
            'model' => $model,
            'cartItems' => $cartItems,
            'totalPrice' => $totalPrice,
        ]);
    }

    public function actionHistory()
    {
        $query = Order::find()
            ->where(['user_id' => Yii::$app->user->id])
            ->orderBy(['created_at' => SORT_DESC]);

        $dataProvider = new ActiveDataProvider([
            'query' => $query,
            'pagination' => [
                'pageSize' => 10,
            ],
        ]);

        return $this->render('history', [
            'dataProvider' => $dataProvider,
        ]);
    }

    public function actionCancel($id)
    {
        $order = Order::find()
            ->where(['id' => $id, 'user_id' => Yii::$app->user->id])
            ->one();

        if (!$order) {
            throw new NotFoundHttpException('Заказ не найден.');
        }

        // Можно отменять только новые или подтвержденные заказы
        if (!$order->isStatusNew() && !$order->isStatusConfirmed()) {
            Yii::$app->session->setFlash('error', 'Невозможно отменить заказ в текущем статусе.');
            return $this->redirect(['view', 'id' => $id]);
        }

        $transaction = Yii::$app->db->beginTransaction();
        try {
            // Возвращаем товары на склад
            foreach ($order->orderItems as $orderItem) {
                $product = $orderItem->product;
                $product->counts += $orderItem->quantity;
                if (!$product->save()) {
                    throw new \Exception('Ошибка возврата товара на склад');
                }
            }

            // Меняем статус заказа
            $order->setStatusToCancelled();
            if (!$order->save()) {
                throw new \Exception('Ошибка сохранения статуса заказа');
            }

            $transaction->commit();
            Yii::$app->session->setFlash('success', 'Заказ успешно отменен.');
        } catch (\Exception $e) {
            $transaction->rollBack();
            Yii::error($e->getMessage(), 'order');
            Yii::$app->session->setFlash('error', 'Ошибка при отмене заказа: ' . $e->getMessage());
        }

        return $this->redirect(['view', 'id' => $id]);
    }

    private function sendOrderConfirmation(Order $order)
    {
        // Отправка email уведомления
        if ($order->user && $order->user->email) {
            try {
                Yii::$app->mailer->compose('order-confirmation', ['order' => $order])
                    ->setFrom([Yii::$app->params['adminEmail'] => Yii::$app->name])
                    ->setTo($order->user->email)
                    ->setSubject('Подтверждение заказа №' . $order->id)
                    ->send();
            } catch (\Exception $e) {
                Yii::error('Ошибка отправки email: ' . $e->getMessage(), 'order');
            }
        }

        // Можно добавить отправку SMS или другие уведомления
    }

    // Для админ-панели (если нужно)
    public function actionAdminIndex()
    {
        if (!Yii::$app->user->can('admin')) {
            throw new \yii\web\ForbiddenHttpException('Доступ запрещен.');
        }

        $searchModel = new \app\models\OrderSearch();
        $dataProvider = $searchModel->search(Yii::$app->request->queryParams);

        return $this->render('admin-index', [
            'searchModel' => $searchModel,
            'dataProvider' => $dataProvider,
        ]);
    }

    public function actionUpdateStatus($id)
    {
        if (!Yii::$app->user->can('admin')) {
            throw new \yii\web\ForbiddenHttpException('Доступ запрещен.');
        }

        $order = Order::findOne($id);
        if (!$order) {
            throw new NotFoundHttpException('Заказ не найден.');
        }

        if ($order->load(Yii::$app->request->post()) && $order->save()) {
            Yii::$app->session->setFlash('success', 'Статус заказа обновлен.');
            return $this->redirect(['admin-index']);
        }

        return $this->render('update-status', [
            'model' => $order,
        ]);
    }
}