<?php

namespace app\controllers;

use Yii;
use app\models\Cart;
use app\models\Product;
use yii\web\Controller;
use yii\web\NotFoundHttpException;
use yii\filters\VerbFilter;
use yii\filters\AccessControl;

class CartController extends Controller
{
    public function behaviors()
    {
        return [
            'access' => [
                'class' => AccessControl::class,
                'rules' => [
                    [
                        'allow' => true,
                        'roles' => ['@'], // Только авторизованные пользователи
                    ],
                ],
            ],
            'verbs' => [
                'class' => VerbFilter::class,
                'actions' => [
                    'delete' => ['POST'],
                    'update' => ['POST'],
                    'add' => ['GET', 'POST'], // ЯВНО разрешаем и POST, и GET
                ],
            ],
        ];
    }

    public function actionAdd()
    {
        $request = Yii::$app->request;
        $userId = Yii::$app->user->id;

        // Получаем ID из разных источников
        if ($request->isPost) {
            $productId = $request->post('product_id') ?: $request->post('id');
        } else {
            // Для GET - из параметров маршрута или GET-параметров
            $productId = $request->get('id');
        }

        $quantity = $request->isPost
            ? $request->post('quantity', 1)
            : $request->get('quantity', 1);

        // Если ID не найден
        if (!$productId) {
            Yii::$app->session->setFlash('error', 'ID товара не найден в запросе.');
            return $this->redirect($request->referrer ?: ['site/index']);
        }

        $product = Product::findOne($productId);
        if (!$product) {
            Yii::$app->session->setFlash('error', 'Товар не найден в базе данных.');
            return $this->redirect($request->referrer ?: ['site/index']);
        }

        if ($product->counts < $quantity) {
            Yii::$app->session->setFlash('error', "Недостаточно товара в наличии. Доступно: {$product->counts} шт.");
            return $this->redirect($request->referrer ?: ['product/view', 'id' => $productId]);
        }

        $cartItem = Cart::find()
            ->where(['user_id' => $userId, 'product_id' => $productId])
            ->one();

        if ($cartItem) {
            $cartItem->quantity += $quantity;
        } else {
            $cartItem = new Cart();
            $cartItem->user_id = $userId;
            $cartItem->product_id = $productId;
            $cartItem->quantity = $quantity;
        }

        if ($cartItem->save()) {
            Yii::$app->session->setFlash('success', 'Товар добавлен в корзину.');
        } else {
            Yii::$app->session->setFlash('error', 'Ошибка при добавлении товара в корзину.');
        }

        return $this->redirect($request->referrer ?: ['cart/index']);
    }

    public function actionIndex()
    {
        $userId = Yii::$app->user->id;
        $cartItems = Cart::find()
            ->where(['user_id' => $userId])
            ->with('product')
            ->all();

        $totalPrice = 0;
        foreach ($cartItems as $item) {
            $totalPrice += $item->quantity * $item->product->price;
        }

        return $this->render('index', [
            'cartItems' => $cartItems,
            'totalPrice' => $totalPrice,
        ]);
    }
    public function actionUpdate()
    {
        $request = Yii::$app->request;
        $id = $request->post('id');
        $quantity = $request->post('quantity');

        $cartItem = Cart::findOne($id);
        if (!$cartItem || $cartItem->user_id != Yii::$app->user->id) {
            throw new NotFoundHttpException('Запись не найдена.');
        }

        if ($quantity < 1) {
            $cartItem->delete();
            Yii::$app->session->setFlash('success', 'Товар удален из корзины.');
        } else {
            if ($cartItem->product->counts < $quantity) {
                Yii::$app->session->setFlash('error', 'Недостаточно товара в наличии.');
                return $this->redirect(['cart/index']);
            }

            $cartItem->quantity = $quantity;
            if ($cartItem->save()) {
                Yii::$app->session->setFlash('success', 'Количество обновлено.');
            } else {
                Yii::$app->session->setFlash('error', 'Ошибка при обновлении количества.');
            }
        }

        return $this->redirect(['cart/index']);
    }

    public function actionDelete($id)
    {
        $cartItem = Cart::findOne($id);
        if (!$cartItem || $cartItem->user_id != Yii::$app->user->id) {
            throw new NotFoundHttpException('Запись не найдена.');
        }

        $cartItem->delete();
        Yii::$app->session->setFlash('success', 'Товар удален из корзины.');

        return $this->redirect(['cart/index']);
    }

    public function actionClear()
    {
        Cart::deleteAll(['user_id' => Yii::$app->user->id]);
        Yii::$app->session->setFlash('success', 'Корзина очищена.');

        return $this->redirect(['cart/index']);
    }

    public function actionRepeatOrder($id)
    {
        $order = \app\models\Order::find()
            ->where(['id' => $id, 'user_id' => Yii::$app->user->id])
            ->with('orderItems.product')
            ->one();

        if (!$order) {
            throw new NotFoundHttpException('Заказ не найден.');
        }

        // Очищаем текущую корзину
        Cart::deleteAll(['user_id' => Yii::$app->user->id]);

        // Добавляем товары из заказа в корзину
        foreach ($order->orderItems as $orderItem) {
            $cartItem = new Cart();
            $cartItem->user_id = Yii::$app->user->id;
            $cartItem->product_id = $orderItem->product_id;
            $cartItem->quantity = $orderItem->quantity;
            $cartItem->save();
        }

        Yii::$app->session->setFlash('success', 'Товары из заказа добавлены в корзину.');
        return $this->redirect(['cart/index']);
    }
}